// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2018 MediaTek Inc.
 *
 * Author: ZH Chen <zh.chen@mediatek.com>
 *
 */

#include <linux/module.h>
#include "pinctrl-mtk-mt6765.h"
#include "pinctrl-paris.h"

/* MT6765 have multiple bases to program pin configuration listed as the below:
 * iocfg[0]:0x10005000, iocfg[1]:0x10002C00, iocfg[2]:0x10002800,
 * iocfg[3]:0x10002A00, iocfg[4]:0x10002000, iocfg[5]:0x10002200,
 * iocfg[6]:0x10002500, iocfg[7]:0x10002600.
 * _i_base could be used to indicate what base the pin should be mapped into.
 */

#define PIN_FIELD_BASE(_s_pin, _e_pin, _i_base, _s_addr, _x_addrs, _s_bit, _x_bits)	\
	PIN_FIELD_CALC(_s_pin, _e_pin, _i_base, _s_addr, _x_addrs, _s_bit,	\
		       _x_bits, 32, 0)

#define PINS_FIELD_BASE(_s_pin, _e_pin, _i_base, _s_addr, _x_addrs, _s_bit, _x_bits)	\
	PIN_FIELD_CALC(_s_pin, _e_pin, _i_base, _s_addr, _x_addrs, _s_bit,	\
		      _x_bits, 32, 1)

static const struct mtk_pin_field_calc mt6765_pin_mode_range[] = {
	PIN_FIELD(0, 202, 0x300, 0x10, 0, 4),
};

static const struct mtk_pin_field_calc mt6765_pin_dir_range[] = {
	PIN_FIELD(0, 202, 0x0, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_di_range[] = {
	PIN_FIELD(0, 202, 0x200, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_do_range[] = {
	PIN_FIELD(0, 202, 0x100, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_smt_range[] = {
	PINS_FIELD_BASE(0, 3, 2, 0x00b0, 0x10, 4, 1),
	PINS_FIELD_BASE(4, 7, 2, 0x00b0, 0x10, 5, 1),
	PIN_FIELD_BASE(8, 8, 3, 0x0080, 0x10, 3, 1),
	PINS_FIELD_BASE(9, 11, 2, 0x00b0, 0x10, 6, 1),
	PIN_FIELD_BASE(12, 12, 5, 0x0060, 0x10, 9, 1),
	PINS_FIELD_BASE(13, 16, 6, 0x00b0, 0x10, 10, 1),
	PINS_FIELD_BASE(17, 20, 6, 0x00b0, 0x10, 8, 1),
	PINS_FIELD_BASE(21, 24, 6, 0x00b0, 0x10, 9, 1),
	PINS_FIELD_BASE(25, 28, 6, 0x00b0, 0x10, 7, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x00b0, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x00b0, 0x10, 1, 1),
	PINS_FIELD_BASE(31, 34, 6, 0x00b0, 0x10, 2, 1),
	PINS_FIELD_BASE(35, 36, 6, 0x00b0, 0x10, 5, 1),
	PIN_FIELD_BASE(37, 37, 6, 0x00b0, 0x10, 6, 1),
	PIN_FIELD_BASE(38, 38, 6, 0x00b0, 0x10, 4, 1),
	PINS_FIELD_BASE(39, 40, 6, 0x00b0, 0x10, 3, 1),
	PINS_FIELD_BASE(41, 42, 7, 0x00c0, 0x10, 6, 1),
	PIN_FIELD_BASE(43, 43, 7, 0x00c0, 0x10, 3, 1),
	PIN_FIELD_BASE(44, 44, 7, 0x00c0, 0x10, 4, 1),
	PIN_FIELD_BASE(45, 45, 7, 0x00c0, 0x10, 8, 1),
	PINS_FIELD_BASE(46, 47, 7, 0x00c0, 0x10, 7, 1),
	PIN_FIELD_BASE(48, 48, 7, 0x00c0, 0x10, 15, 1),
	PIN_FIELD_BASE(49, 49, 7, 0x00c0, 0x10, 17, 1),
	PIN_FIELD_BASE(50, 50, 7, 0x00c0, 0x10, 14, 1),
	PIN_FIELD_BASE(51, 51, 7, 0x00c0, 0x10, 16, 1),
	PINS_FIELD_BASE(52, 57, 7, 0x00c0, 0x10, 0, 1),
	PINS_FIELD_BASE(58, 60, 7, 0x00c0, 0x10, 12, 1),
	PINS_FIELD_BASE(61, 62, 3, 0x0080, 0x10, 5, 1),
	PINS_FIELD_BASE(63, 64, 3, 0x0080, 0x10, 4, 1),
	PINS_FIELD_BASE(65, 66, 3, 0x0080, 0x10, 7, 1),
	PINS_FIELD_BASE(67, 68, 3, 0x0080, 0x10, 6, 1),
	PINS_FIELD_BASE(69, 73, 3, 0x0080, 0x10, 1, 1),
	PINS_FIELD_BASE(74, 78, 3, 0x0080, 0x10, 2, 1),
	PINS_FIELD_BASE(79, 80, 3, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(83, 83, 3, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(85, 85, 7, 0x00c0, 0x10, 12, 1),
	PIN_FIELD_BASE(86, 86, 7, 0x00c0, 0x10, 13, 1),
	PIN_FIELD_BASE(87, 87, 7, 0x00c0, 0x10, 2, 1),
	PIN_FIELD_BASE(88, 88, 7, 0x00c0, 0x10, 1, 1),
	PIN_FIELD_BASE(89, 89, 2, 0x00b0, 0x10, 13, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x0080, 0x10, 8, 1),
	PINS_FIELD_BASE(91, 92, 2, 0x00b0, 0x10, 8, 1),
	PINS_FIELD_BASE(93, 94, 2, 0x00b0, 0x10, 7, 1),
	PINS_FIELD_BASE(95, 96, 2, 0x00b0, 0x10, 14, 1),
	PINS_FIELD_BASE(97, 98, 2, 0x00b0, 0x10, 2, 1),
	PIN_FIELD_BASE(99, 99, 2, 0x00b0, 0x10, 0, 1),
	PIN_FIELD_BASE(100, 100, 2, 0x00b0, 0x10, 1, 1),
	PINS_FIELD_BASE(101, 102, 2, 0x00b0, 0x10, 3, 1),
	PIN_FIELD_BASE(103, 103, 2, 0x00b0, 0x10, 9, 1),
	PIN_FIELD_BASE(104, 104, 2, 0x00b0, 0x10, 11, 1),
	PIN_FIELD_BASE(105, 105, 2, 0x00b0, 0x10, 10, 1),
	PIN_FIELD_BASE(106, 106, 2, 0x00b0, 0x10, 12, 1),
	PIN_FIELD_BASE(107, 107, 1, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(108, 108, 1, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(109, 109, 1, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(110, 110, 1, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(111, 111, 1, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(112, 112, 1, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(113, 113, 1, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(114, 114, 1, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(115, 115, 1, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(116, 116, 1, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(117, 117, 1, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(118, 118, 1, 0x0080, 0x10, 13, 1),
	PIN_FIELD_BASE(119, 119, 1, 0x0080, 0x10, 14, 1),
	PIN_FIELD_BASE(120, 120, 1, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(121, 121, 1, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(122, 122, 4, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(123, 123, 4, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(124, 124, 4, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(127, 127, 4, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(128, 128, 4, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(131, 131, 4, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(132, 132, 4, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(133, 133, 4, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(134, 134, 5, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(135, 135, 5, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(136, 136, 5, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(137, 137, 5, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(138, 138, 5, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(139, 139, 5, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(140, 140, 5, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(141, 141, 5, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(142, 142, 5, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(143, 143, 5, 0x0060, 0x10, 3, 1),
	PINS_FIELD_BASE(144, 147, 5, 0x0060, 0x10, 10, 1),
	PINS_FIELD_BASE(148, 149, 5, 0x0060, 0x10, 12, 1),
	PINS_FIELD_BASE(150, 151, 7, 0x00c0, 0x10, 9, 1),
	PINS_FIELD_BASE(152, 153, 7, 0x00c0, 0x10, 10, 1),
	PIN_FIELD_BASE(154, 154, 7, 0x00c0, 0x10, 11, 1),
	PINS_FIELD_BASE(155, 158, 3, 0x0080, 0x10, 13, 1),
	PIN_FIELD_BASE(159, 159, 7, 0x00c0, 0x10, 11, 1),
	PIN_FIELD_BASE(160, 160, 5, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(161, 161, 1, 0x0080, 0x10, 15, 1),
	PIN_FIELD_BASE(162, 162, 1, 0x0080, 0x10, 16, 1),
	PINS_FIELD_BASE(163, 170, 4, 0x0080, 0x10, 0, 1),
	PINS_FIELD_BASE(171, 179, 7, 0x00c0, 0x10, 5, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_pd_range[] = {
	PIN_FIELD_BASE(0, 0, 2, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(1, 1, 2, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(2, 2, 2, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(3, 3, 2, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(4, 4, 2, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(5, 5, 2, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(6, 6, 2, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(7, 7, 2, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(8, 8, 3, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(9, 9, 2, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(10, 10, 2, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(11, 11, 2, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(12, 12, 5, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(13, 13, 6, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(14, 14, 6, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(15, 15, 6, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(16, 16, 6, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(17, 17, 6, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(18, 18, 6, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(19, 19, 6, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(20, 20, 6, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(21, 21, 6, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(22, 22, 6, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(23, 23, 6, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(24, 24, 6, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(25, 25, 6, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(26, 26, 6, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(27, 27, 6, 0x0040, 0x10, 3, 1),
	PINS_FIELD_BASE(28, 40, 6, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(41, 41, 7, 0x0060, 0x10, 19, 1),
	PIN_FIELD_BASE(42, 42, 7, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(43, 43, 7, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(44, 44, 7, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(45, 45, 7, 0x0060, 0x10, 22, 1),
	PIN_FIELD_BASE(46, 46, 7, 0x0060, 0x10, 21, 1),
	PIN_FIELD_BASE(47, 47, 7, 0x0060, 0x10, 20, 1),
	PIN_FIELD_BASE(48, 48, 7, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(49, 49, 7, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(50, 50, 7, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(51, 51, 7, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(52, 52, 7, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(53, 53, 7, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(54, 54, 7, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(55, 55, 7, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(56, 56, 7, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(57, 57, 7, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(58, 58, 7, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(59, 59, 7, 0x0060, 0x10, 31, 1),
	PIN_FIELD_BASE(60, 60, 7, 0x0060, 0x10, 30, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(63, 63, 3, 0x0040, 0x10, 17, 1),
	PIN_FIELD_BASE(64, 64, 3, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(65, 65, 3, 0x0040, 0x10, 20, 1),
	PIN_FIELD_BASE(66, 66, 3, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(67, 67, 3, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(68, 68, 3, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(69, 69, 3, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(70, 70, 3, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(71, 71, 3, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(72, 72, 3, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(73, 73, 3, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(74, 74, 3, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(75, 75, 3, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(76, 76, 3, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(77, 77, 3, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(78, 78, 3, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(79, 79, 3, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(80, 80, 3, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(83, 83, 3, 0x0040, 0x10, 22, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(85, 85, 7, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(86, 86, 7, 0x0060, 0x10, 29, 1),
	PIN_FIELD_BASE(87, 87, 7, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(88, 88, 7, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(89, 89, 2, 0x0040, 0x10, 21, 1),
	PINS_FIELD_BASE(90, 94, 3, 0x0040, 0x10, 21, 1),
	PIN_FIELD_BASE(95, 95, 2, 0x0040, 0x10, 22, 1),
	PIN_FIELD_BASE(96, 96, 2, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(97, 97, 2, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(98, 98, 2, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(99, 99, 2, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(100, 100, 2, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(101, 101, 2, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(102, 102, 2, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(103, 103, 2, 0x0040, 0x10, 17, 1),
	PIN_FIELD_BASE(104, 104, 2, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(105, 105, 2, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(106, 106, 2, 0x0040, 0x10, 20, 1),
	PIN_FIELD_BASE(107, 107, 1, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(108, 108, 1, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(109, 109, 1, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(110, 110, 1, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(111, 111, 1, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(112, 112, 1, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(113, 113, 1, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(114, 114, 1, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(115, 115, 1, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(116, 116, 1, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(117, 117, 1, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(118, 118, 1, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(119, 119, 1, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(120, 120, 1, 0x0040, 0x10, 11, 1),
	PINS_FIELD_BASE(121, 133, 1, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(134, 134, 5, 0x0030, 0x10, 14, 1),
	PIN_FIELD_BASE(135, 135, 5, 0x0030, 0x10, 19, 1),
	PIN_FIELD_BASE(136, 136, 5, 0x0030, 0x10, 1, 1),
	PIN_FIELD_BASE(137, 137, 5, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(138, 138, 5, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(139, 139, 5, 0x0030, 0x10, 5, 1),
	PIN_FIELD_BASE(140, 140, 5, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(141, 141, 5, 0x0030, 0x10, 6, 1),
	PIN_FIELD_BASE(142, 142, 5, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(143, 143, 5, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(144, 144, 5, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(145, 145, 5, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(146, 146, 5, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(147, 147, 5, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(148, 148, 5, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(149, 149, 5, 0x0030, 0x10, 16, 1),
	PIN_FIELD_BASE(150, 150, 7, 0x0060, 0x10, 23, 1),
	PIN_FIELD_BASE(151, 151, 7, 0x0060, 0x10, 24, 1),
	PIN_FIELD_BASE(152, 152, 7, 0x0060, 0x10, 25, 1),
	PIN_FIELD_BASE(153, 153, 7, 0x0060, 0x10, 26, 1),
	PIN_FIELD_BASE(154, 154, 7, 0x0060, 0x10, 28, 1),
	PIN_FIELD_BASE(155, 155, 3, 0x0040, 0x10, 28, 1),
	PIN_FIELD_BASE(156, 156, 3, 0x0040, 0x10, 27, 1),
	PIN_FIELD_BASE(157, 157, 3, 0x0040, 0x10, 29, 1),
	PIN_FIELD_BASE(158, 158, 3, 0x0040, 0x10, 26, 1),
	PIN_FIELD_BASE(159, 159, 7, 0x0060, 0x10, 27, 1),
	PIN_FIELD_BASE(160, 160, 5, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(161, 161, 1, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(162, 162, 1, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(163, 163, 4, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(164, 164, 4, 0x0020, 0x10, 1, 1),
	PIN_FIELD_BASE(165, 165, 4, 0x0020, 0x10, 2, 1),
	PIN_FIELD_BASE(166, 166, 4, 0x0020, 0x10, 3, 1),
	PIN_FIELD_BASE(167, 167, 4, 0x0020, 0x10, 4, 1),
	PIN_FIELD_BASE(168, 168, 4, 0x0020, 0x10, 5, 1),
	PIN_FIELD_BASE(169, 169, 4, 0x0020, 0x10, 6, 1),
	PIN_FIELD_BASE(170, 170, 4, 0x0020, 0x10, 7, 1),
	PIN_FIELD_BASE(171, 171, 7, 0x0060, 0x10, 17, 1),
	PIN_FIELD_BASE(172, 172, 7, 0x0060, 0x10, 18, 1),
	PIN_FIELD_BASE(173, 173, 7, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(174, 174, 7, 0x0060, 0x10, 12, 1),
	PIN_FIELD_BASE(175, 175, 7, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(176, 176, 7, 0x0060, 0x10, 14, 1),
	PIN_FIELD_BASE(177, 177, 7, 0x0060, 0x10, 15, 1),
	PINS_FIELD_BASE(178, 179, 7, 0x0060, 0x10, 16, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_pu_range[] = {
	PIN_FIELD_BASE(0, 0, 2, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(1, 1, 2, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(2, 2, 2, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(3, 3, 2, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(4, 4, 2, 0x0060, 0x10, 12, 1),
	PIN_FIELD_BASE(5, 5, 2, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(6, 6, 2, 0x0060, 0x10, 14, 1),
	PIN_FIELD_BASE(7, 7, 2, 0x0060, 0x10, 15, 1),
	PIN_FIELD_BASE(8, 8, 3, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(9, 9, 2, 0x0060, 0x10, 16, 1),
	PIN_FIELD_BASE(10, 10, 2, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(11, 11, 2, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(12, 12, 5, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(13, 13, 6, 0x0060, 0x10, 14, 1),
	PIN_FIELD_BASE(14, 14, 6, 0x0060, 0x10, 13, 1),
	PIN_FIELD_BASE(15, 15, 6, 0x0060, 0x10, 15, 1),
	PIN_FIELD_BASE(16, 16, 6, 0x0060, 0x10, 12, 1),
	PIN_FIELD_BASE(17, 17, 6, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(18, 18, 6, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(19, 19, 6, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(20, 20, 6, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(21, 21, 6, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(22, 22, 6, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(23, 23, 6, 0x0060, 0x10, 11, 1),
	PIN_FIELD_BASE(24, 24, 6, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(25, 25, 6, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(26, 26, 6, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(27, 27, 6, 0x0060, 0x10, 3, 1),
	PINS_FIELD_BASE(28, 40, 6, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(41, 41, 7, 0x0080, 0x10, 19, 1),
	PIN_FIELD_BASE(42, 42, 7, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(43, 43, 7, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(44, 44, 7, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(45, 45, 7, 0x0080, 0x10, 22, 1),
	PIN_FIELD_BASE(46, 46, 7, 0x0080, 0x10, 21, 1),
	PIN_FIELD_BASE(47, 47, 7, 0x0080, 0x10, 20, 1),
	PIN_FIELD_BASE(48, 48, 7, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(49, 49, 7, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(50, 50, 7, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(51, 51, 7, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(52, 52, 7, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(53, 53, 7, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(54, 54, 7, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(55, 55, 7, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(56, 56, 7, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(57, 57, 7, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(58, 58, 7, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(59, 59, 7, 0x0080, 0x10, 31, 1),
	PIN_FIELD_BASE(60, 60, 7, 0x0080, 0x10, 30, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0050, 0x10, 18, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(63, 63, 3, 0x0050, 0x10, 17, 1),
	PIN_FIELD_BASE(64, 64, 3, 0x0050, 0x10, 13, 1),
	PIN_FIELD_BASE(65, 65, 3, 0x0050, 0x10, 20, 1),
	PIN_FIELD_BASE(66, 66, 3, 0x0050, 0x10, 16, 1),
	PIN_FIELD_BASE(67, 67, 3, 0x0050, 0x10, 19, 1),
	PIN_FIELD_BASE(68, 68, 3, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(69, 69, 3, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(70, 70, 3, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(71, 71, 3, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(72, 72, 3, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(73, 73, 3, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(74, 74, 3, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(75, 75, 3, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(76, 76, 3, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(77, 77, 3, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(78, 78, 3, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(79, 79, 3, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(80, 80, 3, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x0050, 0x10, 25, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x0050, 0x10, 24, 1),
	PIN_FIELD_BASE(83, 83, 3, 0x0050, 0x10, 22, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x0050, 0x10, 23, 1),
	PIN_FIELD_BASE(85, 85, 7, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(86, 86, 7, 0x0080, 0x10, 29, 1),
	PIN_FIELD_BASE(87, 87, 7, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(88, 88, 7, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(89, 89, 2, 0x0060, 0x10, 21, 1),
	PINS_FIELD_BASE(90, 94, 3, 0x0050, 0x10, 21, 1),
	PIN_FIELD_BASE(95, 95, 2, 0x0060, 0x10, 22, 1),
	PIN_FIELD_BASE(96, 96, 2, 0x0060, 0x10, 23, 1),
	PIN_FIELD_BASE(97, 97, 2, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(98, 98, 2, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(99, 99, 2, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(100, 100, 2, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(101, 101, 2, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(102, 102, 2, 0x0060, 0x10, 5, 1),
	PIN_FIELD_BASE(103, 103, 2, 0x0060, 0x10, 17, 1),
	PIN_FIELD_BASE(104, 104, 2, 0x0060, 0x10, 19, 1),
	PIN_FIELD_BASE(105, 105, 2, 0x0060, 0x10, 18, 1),
	PIN_FIELD_BASE(106, 106, 2, 0x0060, 0x10, 20, 1),
	PIN_FIELD_BASE(107, 107, 1, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(108, 108, 1, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(109, 109, 1, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(110, 110, 1, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(111, 111, 1, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(112, 112, 1, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(113, 113, 1, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(114, 114, 1, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(115, 115, 1, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(116, 116, 1, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(117, 117, 1, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(118, 118, 1, 0x0050, 0x10, 13, 1),
	PIN_FIELD_BASE(119, 119, 1, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(120, 120, 1, 0x0050, 0x10, 11, 1),
	PINS_FIELD_BASE(121, 133, 1, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(134, 134, 5, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(135, 135, 5, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(136, 136, 5, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(137, 137, 5, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(138, 138, 5, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(139, 139, 5, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(140, 140, 5, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(141, 141, 5, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(142, 142, 5, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(143, 143, 5, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(144, 144, 5, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(145, 145, 5, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(146, 146, 5, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(147, 147, 5, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(148, 148, 5, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(149, 149, 5, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(150, 150, 7, 0x0080, 0x10, 23, 1),
	PIN_FIELD_BASE(151, 151, 7, 0x0080, 0x10, 24, 1),
	PIN_FIELD_BASE(152, 152, 7, 0x0080, 0x10, 25, 1),
	PIN_FIELD_BASE(153, 153, 7, 0x0080, 0x10, 26, 1),
	PIN_FIELD_BASE(154, 154, 7, 0x0080, 0x10, 28, 1),
	PIN_FIELD_BASE(155, 155, 3, 0x0050, 0x10, 28, 1),
	PIN_FIELD_BASE(156, 156, 3, 0x0050, 0x10, 27, 1),
	PIN_FIELD_BASE(157, 157, 3, 0x0050, 0x10, 29, 1),
	PIN_FIELD_BASE(158, 158, 3, 0x0050, 0x10, 26, 1),
	PIN_FIELD_BASE(159, 159, 7, 0x0080, 0x10, 27, 1),
	PIN_FIELD_BASE(160, 160, 5, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(161, 161, 1, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(162, 162, 1, 0x0050, 0x10, 16, 1),
	PIN_FIELD_BASE(163, 163, 4, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(164, 164, 4, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(165, 165, 4, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(166, 166, 4, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(167, 167, 4, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(168, 168, 4, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(169, 169, 4, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(170, 170, 4, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(171, 171, 7, 0x0080, 0x10, 17, 1),
	PIN_FIELD_BASE(172, 172, 7, 0x0080, 0x10, 18, 1),
	PIN_FIELD_BASE(173, 173, 7, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(174, 174, 7, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(175, 175, 7, 0x0080, 0x10, 13, 1),
	PIN_FIELD_BASE(176, 176, 7, 0x0080, 0x10, 14, 1),
	PIN_FIELD_BASE(177, 177, 7, 0x0080, 0x10, 15, 1),
	PINS_FIELD_BASE(178, 179, 7, 0x0080, 0x10, 16, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_tdsel_range[] = {
	PINS_FIELD_BASE(0, 3, 2, 0x00c0, 0x10, 16, 4),
	PINS_FIELD_BASE(4, 7, 2, 0x00c0, 0x10, 20, 4),
	PIN_FIELD_BASE(8, 8, 3, 0x0090, 0x10, 12, 4),
	PINS_FIELD_BASE(9, 11, 2, 0x00c0, 0x10, 24, 4),
	PIN_FIELD_BASE(12, 12, 5, 0x0080, 0x10, 4, 4),
	PINS_FIELD_BASE(13, 16, 6, 0x00e0, 0x10, 8, 4),
	PINS_FIELD_BASE(17, 20, 6, 0x00e0, 0x10, 0, 4),
	PINS_FIELD_BASE(21, 24, 6, 0x00e0, 0x10, 4, 4),
	PINS_FIELD_BASE(25, 28, 6, 0x00d0, 0x10, 28, 4),
	PIN_FIELD_BASE(29, 29, 6, 0x00d0, 0x10, 0, 4),
	PIN_FIELD_BASE(30, 30, 6, 0x00d0, 0x10, 4, 4),
	PINS_FIELD_BASE(31, 34, 6, 0x00d0, 0x10, 8, 4),
	PINS_FIELD_BASE(35, 36, 6, 0x00d0, 0x10, 20, 4),
	PIN_FIELD_BASE(37, 37, 6, 0x00d0, 0x10, 24, 4),
	PIN_FIELD_BASE(38, 38, 6, 0x00d0, 0x10, 16, 4),
	PINS_FIELD_BASE(39, 40, 6, 0x00d0, 0x10, 12, 4),
	PINS_FIELD_BASE(41, 42, 7, 0x00d0, 0x10, 24, 4),
	PIN_FIELD_BASE(43, 43, 7, 0x00d0, 0x10, 12, 4),
	PIN_FIELD_BASE(44, 44, 7, 0x00d0, 0x10, 16, 4),
	PIN_FIELD_BASE(45, 45, 7, 0x00e0, 0x10, 0, 4),
	PINS_FIELD_BASE(46, 47, 7, 0x00d0, 0x10, 28, 4),
	PINS_FIELD_BASE(48, 49, 7, 0x00e0, 0x10, 28, 4),
	PINS_FIELD_BASE(50, 51, 7, 0x00e0, 0x10, 24, 4),
	PINS_FIELD_BASE(52, 57, 7, 0x00d0, 0x10, 0, 4),
	PINS_FIELD_BASE(58, 60, 7, 0x00e0, 0x10, 16, 4),
	PINS_FIELD_BASE(61, 62, 3, 0x0090, 0x10, 20, 4),
	PINS_FIELD_BASE(63, 64, 3, 0x0090, 0x10, 16, 4),
	PINS_FIELD_BASE(65, 66, 3, 0x0090, 0x10, 28, 4),
	PINS_FIELD_BASE(67, 68, 3, 0x0090, 0x10, 24, 4),
	PINS_FIELD_BASE(69, 73, 3, 0x0090, 0x10, 4, 4),
	PINS_FIELD_BASE(74, 78, 3, 0x0090, 0x10, 8, 4),
	PINS_FIELD_BASE(79, 80, 3, 0x0090, 0x10, 0, 4),
	PIN_FIELD_BASE(81, 81, 3, 0x00a0, 0x10, 8, 4),
	PINS_FIELD_BASE(82, 83, 3, 0x00a0, 0x10, 4, 4),
	PIN_FIELD_BASE(84, 84, 3, 0x00a0, 0x10, 8, 4),
	PIN_FIELD_BASE(85, 85, 7, 0x00e0, 0x10, 16, 4),
	PIN_FIELD_BASE(86, 86, 7, 0x00e0, 0x10, 20, 4),
	PIN_FIELD_BASE(87, 87, 7, 0x00d0, 0x10, 8, 4),
	PIN_FIELD_BASE(88, 88, 7, 0x00d0, 0x10, 4, 4),
	PIN_FIELD_BASE(89, 89, 2, 0x00d0, 0x10, 12, 4),
	PIN_FIELD_BASE(90, 90, 3, 0x00a0, 0x10, 0, 4),
	PINS_FIELD_BASE(91, 92, 2, 0x00d0, 0x10, 0, 4),
	PINS_FIELD_BASE(93, 94, 2, 0x00c0, 0x10, 28, 4),
	PINS_FIELD_BASE(95, 96, 2, 0x00d0, 0x10, 16, 4),
	PINS_FIELD_BASE(97, 98, 2, 0x00c0, 0x10, 8, 4),
	PIN_FIELD_BASE(99, 99, 2, 0x00c0, 0x10, 0, 4),
	PIN_FIELD_BASE(100, 100, 2, 0x00c0, 0x10, 4, 4),
	PINS_FIELD_BASE(101, 102, 2, 0x00c0, 0x10, 12, 4),
	PINS_FIELD_BASE(103, 104, 2, 0x00d0, 0x10, 4, 4),
	PINS_FIELD_BASE(105, 106, 2, 0x00d0, 0x10, 8, 4),
	PIN_FIELD_BASE(107, 107, 1, 0x0090, 0x10, 16, 4),
	PIN_FIELD_BASE(108, 108, 1, 0x0090, 0x10, 12, 4),
	PIN_FIELD_BASE(109, 109, 1, 0x0090, 0x10, 20, 4),
	PIN_FIELD_BASE(110, 110, 1, 0x0090, 0x10, 0, 4),
	PIN_FIELD_BASE(111, 111, 1, 0x0090, 0x10, 4, 4),
	PIN_FIELD_BASE(112, 112, 1, 0x0090, 0x10, 8, 4),
	PIN_FIELD_BASE(113, 113, 1, 0x00a0, 0x10, 4, 4),
	PIN_FIELD_BASE(114, 114, 1, 0x00a0, 0x10, 8, 4),
	PIN_FIELD_BASE(115, 115, 1, 0x0090, 0x10, 24, 4),
	PIN_FIELD_BASE(116, 116, 1, 0x0090, 0x10, 28, 4),
	PIN_FIELD_BASE(117, 117, 1, 0x00a0, 0x10, 16, 4),
	PIN_FIELD_BASE(118, 118, 1, 0x00a0, 0x10, 20, 4),
	PIN_FIELD_BASE(119, 119, 1, 0x00a0, 0x10, 24, 4),
	PIN_FIELD_BASE(120, 120, 1, 0x00a0, 0x10, 12, 4),
	PIN_FIELD_BASE(121, 121, 1, 0x00a0, 0x10, 0, 4),
	PIN_FIELD_BASE(122, 122, 4, 0x0090, 0x10, 8, 4),
	PIN_FIELD_BASE(123, 123, 4, 0x0090, 0x10, 12, 4),
	PIN_FIELD_BASE(124, 124, 4, 0x0090, 0x10, 4, 4),
	PINS_FIELD_BASE(125, 130, 4, 0x0090, 0x10, 12, 4),
	PIN_FIELD_BASE(131, 131, 4, 0x0090, 0x10, 16, 4),
	PIN_FIELD_BASE(132, 132, 4, 0x0090, 0x10, 12, 4),
	PIN_FIELD_BASE(133, 133, 4, 0x0090, 0x10, 20, 4),
	PIN_FIELD_BASE(134, 134, 5, 0x0080, 0x10, 12, 4),
	PIN_FIELD_BASE(135, 135, 5, 0x0080, 0x10, 20, 4),
	PIN_FIELD_BASE(136, 136, 5, 0x0070, 0x10, 4, 4),
	PIN_FIELD_BASE(137, 137, 5, 0x0070, 0x10, 28, 4),
	PIN_FIELD_BASE(138, 138, 5, 0x0070, 0x10, 16, 4),
	PIN_FIELD_BASE(139, 139, 5, 0x0070, 0x10, 20, 4),
	PIN_FIELD_BASE(140, 140, 5, 0x0070, 0x10, 0, 4),
	PIN_FIELD_BASE(141, 141, 5, 0x0070, 0x10, 24, 4),
	PIN_FIELD_BASE(142, 142, 5, 0x0070, 0x10, 8, 4),
	PIN_FIELD_BASE(143, 143, 5, 0x0070, 0x10, 12, 4),
	PINS_FIELD_BASE(144, 147, 5, 0x0080, 0x10, 8, 4),
	PINS_FIELD_BASE(148, 149, 5, 0x0080, 0x10, 16, 4),
	PINS_FIELD_BASE(150, 151, 7, 0x00e0, 0x10, 4, 4),
	PINS_FIELD_BASE(152, 153, 7, 0x00e0, 0x10, 8, 4),
	PIN_FIELD_BASE(154, 154, 7, 0x00e0, 0x10, 12, 4),
	PINS_FIELD_BASE(155, 158, 3, 0x00a0, 0x10, 12, 4),
	PIN_FIELD_BASE(159, 159, 7, 0x00e0, 0x10, 12, 4),
	PIN_FIELD_BASE(160, 160, 5, 0x0080, 0x10, 0, 4),
	PINS_FIELD_BASE(161, 162, 1, 0x00a0, 0x10, 28, 4),
	PINS_FIELD_BASE(163, 170, 4, 0x0090, 0x10, 0, 4),
	PINS_FIELD_BASE(171, 179, 7, 0x00d0, 0x10, 20, 4),
};

static const struct mtk_pin_field_calc mt6765_pin_rdsel_range[] = {
	PINS_FIELD_BASE(0, 3, 2, 0x0090, 0x10, 8, 2),
	PINS_FIELD_BASE(4, 7, 2, 0x0090, 0x10, 10, 2),
	PIN_FIELD_BASE(8, 8, 3, 0x0060, 0x10, 6, 2),
	PINS_FIELD_BASE(9, 11, 2, 0x0090, 0x10, 12, 2),
	PIN_FIELD_BASE(12, 12, 5, 0x0050, 0x10, 18, 2),
	PINS_FIELD_BASE(13, 16, 6, 0x00a0, 0x10, 18, 2),
	PINS_FIELD_BASE(17, 20, 6, 0x00a0, 0x10, 14, 2),
	PINS_FIELD_BASE(21, 24, 6, 0x00a0, 0x10, 16, 2),
	PINS_FIELD_BASE(25, 28, 6, 0x00a0, 0x10, 12, 2),
	PIN_FIELD_BASE(29, 29, 6, 0x0090, 0x10, 0, 6),
	PIN_FIELD_BASE(30, 30, 6, 0x0090, 0x10, 6, 6),
	PINS_FIELD_BASE(31, 34, 6, 0x0090, 0x10, 12, 6),
	PINS_FIELD_BASE(35, 36, 6, 0x00a0, 0x10, 0, 6),
	PIN_FIELD_BASE(37, 37, 6, 0x00a0, 0x10, 6, 6),
	PIN_FIELD_BASE(38, 38, 6, 0x0090, 0x10, 24, 6),
	PINS_FIELD_BASE(39, 40, 6, 0x0090, 0x10, 18, 6),
	PINS_FIELD_BASE(41, 42, 7, 0x00a0, 0x10, 12, 2),
	PIN_FIELD_BASE(43, 43, 7, 0x00a0, 0x10, 6, 2),
	PIN_FIELD_BASE(44, 44, 7, 0x00a0, 0x10, 8, 2),
	PIN_FIELD_BASE(45, 45, 7, 0x00a0, 0x10, 16, 2),
	PINS_FIELD_BASE(46, 47, 7, 0x00a0, 0x10, 14, 2),
	PINS_FIELD_BASE(48, 49, 7, 0x00a0, 0x10, 30, 2),
	PINS_FIELD_BASE(50, 51, 7, 0x00a0, 0x10, 28, 2),
	PINS_FIELD_BASE(52, 57, 7, 0x00a0, 0x10, 0, 2),
	PINS_FIELD_BASE(58, 60, 7, 0x00a0, 0x10, 24, 2),
	PINS_FIELD_BASE(61, 62, 3, 0x0060, 0x10, 10, 2),
	PINS_FIELD_BASE(63, 64, 3, 0x0060, 0x10, 8, 2),
	PINS_FIELD_BASE(65, 66, 3, 0x0060, 0x10, 14, 2),
	PINS_FIELD_BASE(67, 68, 3, 0x0060, 0x10, 12, 2),
	PINS_FIELD_BASE(69, 73, 3, 0x0060, 0x10, 2, 2),
	PINS_FIELD_BASE(74, 78, 3, 0x0060, 0x10, 4, 2),
	PINS_FIELD_BASE(79, 80, 3, 0x0060, 0x10, 0, 2),
	PIN_FIELD_BASE(81, 81, 3, 0x0060, 0x10, 20, 2),
	PINS_FIELD_BASE(82, 83, 3, 0x0060, 0x10, 18, 2),
	PIN_FIELD_BASE(84, 84, 3, 0x0060, 0x10, 20, 2),
	PIN_FIELD_BASE(85, 85, 7, 0x00a0, 0x10, 24, 2),
	PIN_FIELD_BASE(86, 86, 7, 0x00a0, 0x10, 26, 2),
	PIN_FIELD_BASE(87, 87, 7, 0x00a0, 0x10, 4, 2),
	PIN_FIELD_BASE(88, 88, 7, 0x00a0, 0x10, 2, 2),
	PIN_FIELD_BASE(89, 89, 2, 0x0090, 0x10, 22, 2),
	PIN_FIELD_BASE(90, 90, 3, 0x0060, 0x10, 16, 2),
	PINS_FIELD_BASE(91, 92, 2, 0x0090, 0x10, 16, 2),
	PINS_FIELD_BASE(93, 94, 2, 0x0090, 0x10, 14, 2),
	PINS_FIELD_BASE(95, 96, 2, 0x0090, 0x10, 24, 2),
	PINS_FIELD_BASE(97, 98, 2, 0x0090, 0x10, 4, 2),
	PIN_FIELD_BASE(99, 99, 2, 0x0090, 0x10, 0, 2),
	PIN_FIELD_BASE(100, 100, 2, 0x0090, 0x10, 2, 2),
	PINS_FIELD_BASE(101, 102, 2, 0x0090, 0x10, 6, 2),
	PINS_FIELD_BASE(103, 104, 2, 0x0090, 0x10, 18, 2),
	PINS_FIELD_BASE(105, 106, 2, 0x0090, 0x10, 20, 2),
	PIN_FIELD_BASE(107, 107, 1, 0x0060, 0x10, 8, 2),
	PIN_FIELD_BASE(108, 108, 1, 0x0060, 0x10, 6, 2),
	PIN_FIELD_BASE(109, 109, 1, 0x0060, 0x10, 10, 2),
	PIN_FIELD_BASE(110, 110, 1, 0x0060, 0x10, 0, 2),
	PIN_FIELD_BASE(111, 111, 1, 0x0060, 0x10, 2, 2),
	PIN_FIELD_BASE(112, 112, 1, 0x0060, 0x10, 4, 2),
	PIN_FIELD_BASE(113, 113, 1, 0x0060, 0x10, 18, 2),
	PIN_FIELD_BASE(114, 114, 1, 0x0060, 0x10, 20, 2),
	PIN_FIELD_BASE(115, 115, 1, 0x0060, 0x10, 12, 2),
	PIN_FIELD_BASE(116, 116, 1, 0x0060, 0x10, 14, 2),
	PIN_FIELD_BASE(117, 117, 1, 0x0060, 0x10, 24, 2),
	PIN_FIELD_BASE(118, 118, 1, 0x0060, 0x10, 26, 2),
	PIN_FIELD_BASE(119, 119, 1, 0x0060, 0x10, 28, 2),
	PIN_FIELD_BASE(120, 120, 1, 0x0060, 0x10, 22, 2),
	PIN_FIELD_BASE(121, 121, 1, 0x0060, 0x10, 16, 2),
	PIN_FIELD_BASE(122, 122, 4, 0x0070, 0x10, 8, 6),
	PIN_FIELD_BASE(123, 123, 4, 0x0070, 0x10, 14, 6),
	PIN_FIELD_BASE(124, 124, 4, 0x0070, 0x10, 2, 6),
	PINS_FIELD_BASE(125, 130, 4, 0x0070, 0x10, 14, 6),
	PIN_FIELD_BASE(131, 131, 4, 0x0070, 0x10, 20, 6),
	PIN_FIELD_BASE(132, 132, 4, 0x0070, 0x10, 14, 6),
	PIN_FIELD_BASE(133, 133, 4, 0x0070, 0x10, 26, 6),
	PIN_FIELD_BASE(134, 134, 5, 0x0050, 0x10, 22, 2),
	PIN_FIELD_BASE(135, 135, 5, 0x0050, 0x10, 30, 2),
	PIN_FIELD_BASE(136, 136, 5, 0x0050, 0x10, 2, 2),
	PIN_FIELD_BASE(137, 137, 5, 0x0050, 0x10, 14, 2),
	PIN_FIELD_BASE(138, 138, 5, 0x0050, 0x10, 8, 2),
	PIN_FIELD_BASE(139, 139, 5, 0x0050, 0x10, 10, 2),
	PIN_FIELD_BASE(140, 140, 5, 0x0050, 0x10, 0, 2),
	PIN_FIELD_BASE(141, 141, 5, 0x0050, 0x10, 12, 2),
	PIN_FIELD_BASE(142, 142, 5, 0x0050, 0x10, 4, 2),
	PIN_FIELD_BASE(143, 143, 5, 0x0050, 0x10, 6, 2),
	PINS_FIELD_BASE(144, 147, 5, 0x0050, 0x10, 20, 2),
	PINS_FIELD_BASE(148, 149, 5, 0x0050, 0x10, 24, 2),
	PINS_FIELD_BASE(150, 151, 7, 0x00a0, 0x10, 18, 2),
	PINS_FIELD_BASE(152, 153, 7, 0x00a0, 0x10, 20, 2),
	PIN_FIELD_BASE(154, 154, 7, 0x00a0, 0x10, 22, 2),
	PINS_FIELD_BASE(155, 158, 3, 0x0060, 0x10, 22, 2),
	PIN_FIELD_BASE(159, 159, 7, 0x00a0, 0x10, 22, 2),
	PIN_FIELD_BASE(160, 160, 5, 0x0050, 0x10, 16, 2),
	PINS_FIELD_BASE(161, 162, 1, 0x0060, 0x10, 30, 2),
	PINS_FIELD_BASE(163, 170, 4, 0x0070, 0x10, 0, 2),
	PINS_FIELD_BASE(171, 179, 7, 0x00a0, 0x10, 10, 2),
};

static const struct mtk_pin_field_calc mt6765_pin_drv_range[] = {
	PINS_FIELD_BASE(0, 2, 2, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(3, 3, 2, 0x0000, 0x10, 15, 3),
	PINS_FIELD_BASE(4, 6, 2, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(7, 7, 2, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(8, 8, 3, 0x0000, 0x10, 9, 3),
	PINS_FIELD_BASE(9, 11, 2, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(12, 12, 5, 0x0000, 0x10, 27, 3),
	PINS_FIELD_BASE(13, 15, 6, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(16, 16, 6, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(17, 17, 6, 0x0000, 0x10, 23, 3),
	PIN_FIELD_BASE(18, 18, 6, 0x0000, 0x10, 26, 3),
	PINS_FIELD_BASE(19, 20, 6, 0x0000, 0x10, 23, 3),
	PINS_FIELD_BASE(21, 23, 6, 0x0000, 0x10, 29, 3),
	PIN_FIELD_BASE(24, 24, 6, 0x0010, 0x10, 0, 3),
	PINS_FIELD_BASE(25, 27, 6, 0x0000, 0x10, 17, 3),
	PIN_FIELD_BASE(28, 28, 6, 0x0000, 0x10, 20, 3),
	PIN_FIELD_BASE(29, 29, 6, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(30, 30, 6, 0x0000, 0x10, 3, 3),
	PINS_FIELD_BASE(31, 34, 6, 0x0000, 0x10, 6, 3),
	PINS_FIELD_BASE(35, 36, 6, 0x0000, 0x10, 13, 2),
	PIN_FIELD_BASE(37, 37, 6, 0x0000, 0x10, 15, 2),
	PIN_FIELD_BASE(38, 38, 6, 0x0000, 0x10, 11, 2),
	PINS_FIELD_BASE(39, 40, 6, 0x0000, 0x10, 9, 2),
	PINS_FIELD_BASE(41, 42, 7, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(43, 43, 7, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(44, 44, 7, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(45, 45, 7, 0x0000, 0x10, 27, 3),
	PINS_FIELD_BASE(46, 47, 7, 0x0000, 0x10, 24, 3),
	PINS_FIELD_BASE(48, 49, 7, 0x0010, 0x10, 18, 3),
	PINS_FIELD_BASE(50, 51, 7, 0x0010, 0x10, 15, 3),
	PINS_FIELD_BASE(52, 57, 7, 0x0000, 0x10, 0, 3),
	PINS_FIELD_BASE(58, 60, 7, 0x0010, 0x10, 9, 3),
	PINS_FIELD_BASE(61, 62, 3, 0x0000, 0x10, 15, 3),
	PINS_FIELD_BASE(63, 64, 3, 0x0000, 0x10, 12, 3),
	PINS_FIELD_BASE(65, 66, 3, 0x0000, 0x10, 21, 3),
	PINS_FIELD_BASE(67, 68, 3, 0x0000, 0x10, 18, 3),
	PINS_FIELD_BASE(69, 73, 3, 0x0000, 0x10, 3, 3),
	PINS_FIELD_BASE(74, 78, 3, 0x0000, 0x10, 6, 3),
	PINS_FIELD_BASE(79, 80, 3, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(81, 81, 3, 0x0010, 0x10, 0, 3),
	PINS_FIELD_BASE(82, 83, 3, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(84, 84, 3, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(85, 85, 7, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(86, 86, 7, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(87, 87, 7, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(88, 88, 7, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(89, 89, 2, 0x0010, 0x10, 15, 3),
	PIN_FIELD_BASE(90, 90, 3, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(91, 91, 2, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(92, 92, 2, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(93, 93, 2, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(94, 94, 2, 0x0010, 0x10, 0, 3),
	PINS_FIELD_BASE(95, 96, 2, 0x0010, 0x10, 18, 3),
	PINS_FIELD_BASE(97, 98, 2, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(99, 99, 2, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(100, 100, 2, 0x0000, 0x10, 3, 3),
	PINS_FIELD_BASE(101, 102, 2, 0x0000, 0x10, 9, 3),
	PINS_FIELD_BASE(103, 104, 2, 0x0010, 0x10, 9, 3),
	PINS_FIELD_BASE(105, 106, 2, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(107, 107, 1, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(108, 108, 1, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(109, 109, 1, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(110, 110, 1, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(111, 111, 1, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(112, 112, 1, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(113, 113, 1, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(114, 114, 1, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(115, 115, 1, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(116, 116, 1, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(117, 117, 1, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(118, 118, 1, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(119, 119, 1, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(120, 120, 1, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(121, 121, 1, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(122, 122, 4, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(123, 123, 4, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(124, 124, 4, 0x0000, 0x10, 6, 3),
	PINS_FIELD_BASE(125, 130, 4, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(131, 131, 4, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(132, 132, 4, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(133, 133, 4, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(134, 134, 5, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(135, 135, 5, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(136, 136, 5, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(137, 137, 5, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(138, 138, 5, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(139, 139, 5, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(140, 140, 5, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(141, 141, 5, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(142, 142, 5, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(143, 143, 5, 0x0000, 0x10, 9, 3),
	PINS_FIELD_BASE(144, 146, 5, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(147, 147, 5, 0x0010, 0x10, 3, 3),
	PINS_FIELD_BASE(148, 149, 5, 0x0010, 0x10, 9, 3),
	PINS_FIELD_BASE(150, 151, 7, 0x0010, 0x10, 0, 3),
	PINS_FIELD_BASE(152, 153, 7, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(154, 154, 7, 0x0010, 0x10, 6, 3),
	PINS_FIELD_BASE(155, 157, 3, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(158, 158, 3, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(159, 159, 7, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(160, 160, 5, 0x0000, 0x10, 24, 3),
	PINS_FIELD_BASE(161, 162, 1, 0x0010, 0x10, 15, 3),
	PINS_FIELD_BASE(163, 166, 4, 0x0000, 0x10, 0, 3),
	PINS_FIELD_BASE(167, 170, 4, 0x0000, 0x10, 3, 3),
	PINS_FIELD_BASE(171, 174, 7, 0x0000, 0x10, 18, 3),
	PINS_FIELD_BASE(175, 179, 7, 0x0000, 0x10, 15, 3),
};

static const struct mtk_pin_field_calc mt6765_pin_pupd_range[] = {
	PINS_FIELD_BASE(0, 28, 0, 0x0050, 0x10, 18, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(32, 32, 6, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(33, 33, 6, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(34, 34, 6, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(35, 35, 6, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(36, 36, 6, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(37, 37, 6, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(38, 38, 6, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(39, 39, 6, 0x0050, 0x10, 8, 1),
	PINS_FIELD_BASE(40, 90, 6, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(91, 91, 2, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(92, 92, 2, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(93, 93, 2, 0x0050, 0x10, 0, 1),
	PINS_FIELD_BASE(94, 121, 2, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(122, 122, 4, 0x0030, 0x10, 1, 1),
	PIN_FIELD_BASE(123, 123, 4, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(124, 124, 4, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x0030, 0x10, 6, 1),
	PIN_FIELD_BASE(127, 127, 4, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(128, 128, 4, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(131, 131, 4, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(132, 132, 4, 0x0030, 0x10, 5, 1),
	PINS_FIELD_BASE(133, 179, 4, 0x0030, 0x10, 11, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_r0_range[] = {
	PINS_FIELD_BASE(0, 28, 4, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(32, 32, 6, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(33, 33, 6, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(34, 34, 6, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(35, 35, 6, 0x0070, 0x10, 10, 1),
	PIN_FIELD_BASE(36, 36, 6, 0x0070, 0x10, 11, 1),
	PIN_FIELD_BASE(37, 37, 6, 0x0070, 0x10, 9, 1),
	PIN_FIELD_BASE(38, 38, 6, 0x0070, 0x10, 6, 1),
	PIN_FIELD_BASE(39, 39, 6, 0x0070, 0x10, 8, 1),
	PINS_FIELD_BASE(40, 90, 6, 0x0070, 0x10, 7, 1),
	PIN_FIELD_BASE(91, 91, 2, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(92, 92, 2, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(93, 93, 2, 0x0070, 0x10, 0, 1),
	PINS_FIELD_BASE(94, 121, 2, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(122, 122, 4, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(123, 123, 4, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(124, 124, 4, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(127, 127, 4, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(128, 128, 4, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(131, 131, 4, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(132, 132, 4, 0x0050, 0x10, 5, 1),
	PINS_FIELD_BASE(133, 179, 4, 0x0050, 0x10, 11, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_r1_range[] = {
	PINS_FIELD_BASE(0, 28, 4, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(32, 32, 6, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(33, 33, 6, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(34, 34, 6, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(35, 35, 6, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(36, 36, 6, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(37, 37, 6, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(38, 38, 6, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(39, 39, 6, 0x0080, 0x10, 8, 1),
	PINS_FIELD_BASE(40, 90, 6, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(91, 91, 2, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(92, 92, 2, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(93, 93, 2, 0x0080, 0x10, 0, 1),
	PINS_FIELD_BASE(94, 121, 2, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(122, 122, 4, 0x0060, 0x10, 1, 1),
	PIN_FIELD_BASE(123, 123, 4, 0x0060, 0x10, 2, 1),
	PIN_FIELD_BASE(124, 124, 4, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x0060, 0x10, 4, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x0060, 0x10, 6, 1),
	PIN_FIELD_BASE(127, 127, 4, 0x0060, 0x10, 8, 1),
	PIN_FIELD_BASE(128, 128, 4, 0x0060, 0x10, 3, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x0060, 0x10, 7, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x0060, 0x10, 9, 1),
	PIN_FIELD_BASE(131, 131, 4, 0x0060, 0x10, 10, 1),
	PIN_FIELD_BASE(132, 132, 4, 0x0060, 0x10, 5, 1),
	PINS_FIELD_BASE(133, 179, 4, 0x0060, 0x10, 11, 1),
};

static const struct mtk_pin_field_calc mt6765_pin_ies_range[] = {
	PIN_FIELD_BASE(0, 0, 2, 0x0030, 0x10, 6, 1),
	PIN_FIELD_BASE(1, 1, 2, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(2, 2, 2, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(3, 3, 2, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(4, 4, 2, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(5, 5, 2, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(6, 6, 2, 0x0030, 0x10, 14, 1),
	PIN_FIELD_BASE(7, 7, 2, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(8, 8, 3, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(9, 9, 2, 0x0030, 0x10, 16, 1),
	PIN_FIELD_BASE(10, 10, 2, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(11, 11, 2, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(12, 12, 5, 0x0020, 0x10, 9, 1),
	PIN_FIELD_BASE(13, 13, 6, 0x0020, 0x10, 26, 1),
	PIN_FIELD_BASE(14, 14, 6, 0x0020, 0x10, 25, 1),
	PIN_FIELD_BASE(15, 15, 6, 0x0020, 0x10, 27, 1),
	PIN_FIELD_BASE(16, 16, 6, 0x0020, 0x10, 24, 1),
	PIN_FIELD_BASE(17, 17, 6, 0x0020, 0x10, 19, 1),
	PIN_FIELD_BASE(18, 18, 6, 0x0020, 0x10, 16, 1),
	PIN_FIELD_BASE(19, 19, 6, 0x0020, 0x10, 18, 1),
	PIN_FIELD_BASE(20, 20, 6, 0x0020, 0x10, 17, 1),
	PIN_FIELD_BASE(21, 21, 6, 0x0020, 0x10, 22, 1),
	PIN_FIELD_BASE(22, 22, 6, 0x0020, 0x10, 21, 1),
	PIN_FIELD_BASE(23, 23, 6, 0x0020, 0x10, 23, 1),
	PIN_FIELD_BASE(24, 24, 6, 0x0020, 0x10, 20, 1),
	PIN_FIELD_BASE(25, 25, 6, 0x0020, 0x10, 14, 1),
	PIN_FIELD_BASE(26, 26, 6, 0x0020, 0x10, 13, 1),
	PIN_FIELD_BASE(27, 27, 6, 0x0020, 0x10, 15, 1),
	PIN_FIELD_BASE(28, 28, 6, 0x0020, 0x10, 12, 1),
	PIN_FIELD_BASE(29, 29, 6, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(30, 30, 6, 0x0020, 0x10, 1, 1),
	PIN_FIELD_BASE(31, 31, 6, 0x0020, 0x10, 5, 1),
	PIN_FIELD_BASE(32, 32, 6, 0x0020, 0x10, 2, 1),
	PIN_FIELD_BASE(33, 33, 6, 0x0020, 0x10, 4, 1),
	PIN_FIELD_BASE(34, 34, 6, 0x0020, 0x10, 3, 1),
	PIN_FIELD_BASE(35, 35, 6, 0x0020, 0x10, 10, 1),
	PIN_FIELD_BASE(36, 36, 6, 0x0020, 0x10, 11, 1),
	PIN_FIELD_BASE(37, 37, 6, 0x0020, 0x10, 9, 1),
	PIN_FIELD_BASE(38, 38, 6, 0x0020, 0x10, 6, 1),
	PIN_FIELD_BASE(39, 39, 6, 0x0020, 0x10, 8, 1),
	PIN_FIELD_BASE(40, 40, 6, 0x0020, 0x10, 7, 1),
	PIN_FIELD_BASE(41, 41, 7, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(42, 42, 7, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(43, 43, 7, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(44, 44, 7, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(45, 45, 7, 0x0040, 0x10, 22, 1),
	PIN_FIELD_BASE(46, 46, 7, 0x0040, 0x10, 21, 1),
	PIN_FIELD_BASE(47, 47, 7, 0x0040, 0x10, 20, 1),
	PIN_FIELD_BASE(48, 48, 7, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(49, 49, 7, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(50, 50, 7, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(51, 51, 7, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(52, 52, 7, 0x0040, 0x10, 1, 1),
	PIN_FIELD_BASE(53, 53, 7, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(54, 54, 7, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(55, 55, 7, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(56, 56, 7, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(57, 57, 7, 0x0040, 0x10, 2, 1),
	PIN_FIELD_BASE(58, 58, 7, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(59, 59, 7, 0x0040, 0x10, 31, 1),
	PIN_FIELD_BASE(60, 60, 7, 0x0040, 0x10, 30, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0030, 0x10, 18, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0030, 0x10, 14, 1),
	PIN_FIELD_BASE(63, 63, 3, 0x0030, 0x10, 17, 1),
	PIN_FIELD_BASE(64, 64, 3, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(65, 65, 3, 0x0030, 0x10, 20, 1),
	PIN_FIELD_BASE(66, 66, 3, 0x0030, 0x10, 16, 1),
	PIN_FIELD_BASE(67, 67, 3, 0x0030, 0x10, 19, 1),
	PIN_FIELD_BASE(68, 68, 3, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(69, 69, 3, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(70, 70, 3, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(71, 71, 3, 0x0030, 0x10, 6, 1),
	PIN_FIELD_BASE(72, 72, 3, 0x0030, 0x10, 5, 1),
	PIN_FIELD_BASE(73, 73, 3, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(74, 74, 3, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(75, 75, 3, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(76, 76, 3, 0x0030, 0x10, 1, 1),
	PIN_FIELD_BASE(77, 77, 3, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(78, 78, 3, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(79, 79, 3, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(80, 80, 3, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x0030, 0x10, 25, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x0030, 0x10, 24, 1),
	PIN_FIELD_BASE(83, 83, 3, 0x0030, 0x10, 22, 1),
	PIN_FIELD_BASE(84, 84, 3, 0x0030, 0x10, 23, 1),
	PIN_FIELD_BASE(85, 85, 7, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(86, 86, 7, 0x0040, 0x10, 29, 1),
	PIN_FIELD_BASE(87, 87, 7, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(88, 88, 7, 0x0040, 0x10, 6, 1),
	PIN_FIELD_BASE(89, 89, 2, 0x0030, 0x10, 25, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x0030, 0x10, 21, 1),
	PIN_FIELD_BASE(91, 91, 2, 0x0030, 0x10, 20, 1),
	PIN_FIELD_BASE(92, 92, 2, 0x0030, 0x10, 19, 1),
	PIN_FIELD_BASE(93, 93, 2, 0x0030, 0x10, 17, 1),
	PIN_FIELD_BASE(94, 94, 2, 0x0030, 0x10, 18, 1),
	PIN_FIELD_BASE(95, 95, 2, 0x0030, 0x10, 26, 1),
	PIN_FIELD_BASE(96, 96, 2, 0x0030, 0x10, 27, 1),
	PIN_FIELD_BASE(97, 97, 2, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(98, 98, 2, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(99, 99, 2, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(100, 100, 2, 0x0030, 0x10, 1, 1),
	PIN_FIELD_BASE(101, 101, 2, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(102, 102, 2, 0x0030, 0x10, 5, 1),
	PIN_FIELD_BASE(103, 103, 2, 0x0030, 0x10, 21, 1),
	PIN_FIELD_BASE(104, 104, 2, 0x0030, 0x10, 23, 1),
	PIN_FIELD_BASE(105, 105, 2, 0x0030, 0x10, 22, 1),
	PIN_FIELD_BASE(106, 106, 2, 0x0030, 0x10, 24, 1),
	PIN_FIELD_BASE(107, 107, 1, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(108, 108, 1, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(109, 109, 1, 0x0030, 0x10, 5, 1),
	PIN_FIELD_BASE(110, 110, 1, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(111, 111, 1, 0x0030, 0x10, 1, 1),
	PIN_FIELD_BASE(112, 112, 1, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(113, 113, 1, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(114, 114, 1, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(115, 115, 1, 0x0030, 0x10, 6, 1),
	PIN_FIELD_BASE(116, 116, 1, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(117, 117, 1, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(118, 118, 1, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(119, 119, 1, 0x0030, 0x10, 14, 1),
	PIN_FIELD_BASE(120, 120, 1, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(121, 121, 1, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(122, 122, 4, 0x0010, 0x10, 9, 1),
	PIN_FIELD_BASE(123, 123, 4, 0x0010, 0x10, 10, 1),
	PIN_FIELD_BASE(124, 124, 4, 0x0010, 0x10, 8, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x0010, 0x10, 12, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x0010, 0x10, 14, 1),
	PIN_FIELD_BASE(127, 127, 4, 0x0010, 0x10, 16, 1),
	PIN_FIELD_BASE(128, 128, 4, 0x0010, 0x10, 11, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x0010, 0x10, 15, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x0010, 0x10, 17, 1),
	PIN_FIELD_BASE(131, 131, 4, 0x0010, 0x10, 18, 1),
	PIN_FIELD_BASE(132, 132, 4, 0x0010, 0x10, 13, 1),
	PIN_FIELD_BASE(133, 133, 4, 0x0010, 0x10, 19, 1),
	PIN_FIELD_BASE(134, 134, 5, 0x0020, 0x10, 14, 1),
	PIN_FIELD_BASE(135, 135, 5, 0x0020, 0x10, 17, 1),
	PIN_FIELD_BASE(136, 136, 5, 0x0020, 0x10, 1, 1),
	PIN_FIELD_BASE(137, 137, 5, 0x0020, 0x10, 7, 1),
	PIN_FIELD_BASE(138, 138, 5, 0x0020, 0x10, 4, 1),
	PIN_FIELD_BASE(139, 139, 5, 0x0020, 0x10, 5, 1),
	PIN_FIELD_BASE(140, 140, 5, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(141, 141, 5, 0x0020, 0x10, 6, 1),
	PIN_FIELD_BASE(142, 142, 5, 0x0020, 0x10, 2, 1),
	PIN_FIELD_BASE(143, 143, 5, 0x0020, 0x10, 3, 1),
	PIN_FIELD_BASE(144, 144, 5, 0x0020, 0x10, 12, 1),
	PIN_FIELD_BASE(145, 145, 5, 0x0020, 0x10, 11, 1),
	PIN_FIELD_BASE(146, 146, 5, 0x0020, 0x10, 13, 1),
	PIN_FIELD_BASE(147, 147, 5, 0x0020, 0x10, 10, 1),
	PIN_FIELD_BASE(148, 148, 5, 0x0020, 0x10, 15, 1),
	PIN_FIELD_BASE(149, 149, 5, 0x0020, 0x10, 16, 1),
	PIN_FIELD_BASE(150, 150, 7, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(151, 151, 7, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(152, 152, 7, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(153, 153, 7, 0x0040, 0x10, 26, 1),
	PIN_FIELD_BASE(154, 154, 7, 0x0040, 0x10, 28, 1),
	PIN_FIELD_BASE(155, 155, 3, 0x0030, 0x10, 28, 1),
	PIN_FIELD_BASE(156, 156, 3, 0x0030, 0x10, 27, 1),
	PIN_FIELD_BASE(157, 157, 3, 0x0030, 0x10, 29, 1),
	PIN_FIELD_BASE(158, 158, 3, 0x0030, 0x10, 26, 1),
	PIN_FIELD_BASE(159, 159, 7, 0x0040, 0x10, 27, 1),
	PIN_FIELD_BASE(160, 160, 5, 0x0020, 0x10, 8, 1),
	PIN_FIELD_BASE(161, 161, 1, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(162, 162, 1, 0x0030, 0x10, 16, 1),
	PIN_FIELD_BASE(163, 163, 4, 0x0010, 0x10, 0, 1),
	PIN_FIELD_BASE(164, 164, 4, 0x0010, 0x10, 1, 1),
	PIN_FIELD_BASE(165, 165, 4, 0x0010, 0x10, 2, 1),
	PIN_FIELD_BASE(166, 166, 4, 0x0010, 0x10, 3, 1),
	PIN_FIELD_BASE(167, 167, 4, 0x0010, 0x10, 4, 1),
	PIN_FIELD_BASE(168, 168, 4, 0x0010, 0x10, 5, 1),
	PIN_FIELD_BASE(169, 169, 4, 0x0010, 0x10, 6, 1),
	PIN_FIELD_BASE(170, 170, 4, 0x0010, 0x10, 7, 1),
	PIN_FIELD_BASE(171, 171, 7, 0x0040, 0x10, 17, 1),
	PIN_FIELD_BASE(172, 172, 7, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(173, 173, 7, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(174, 174, 7, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(175, 175, 7, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(176, 176, 7, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(177, 177, 7, 0x0040, 0x10, 15, 1),
	PINS_FIELD_BASE(178, 179, 7, 0x0040, 0x10, 16, 1),
};

static const struct mtk_pin_reg_calc mt6765_reg_cals[PINCTRL_PIN_REG_MAX] = {
	[PINCTRL_PIN_REG_MODE] = MTK_RANGE(mt6765_pin_mode_range),
	[PINCTRL_PIN_REG_DIR] = MTK_RANGE(mt6765_pin_dir_range),
	[PINCTRL_PIN_REG_DI] = MTK_RANGE(mt6765_pin_di_range),
	[PINCTRL_PIN_REG_DO] = MTK_RANGE(mt6765_pin_do_range),
	[PINCTRL_PIN_REG_SMT] = MTK_RANGE(mt6765_pin_smt_range),
	[PINCTRL_PIN_REG_PD] = MTK_RANGE(mt6765_pin_pd_range),
	[PINCTRL_PIN_REG_PU] = MTK_RANGE(mt6765_pin_pu_range),
	[PINCTRL_PIN_REG_TDSEL] = MTK_RANGE(mt6765_pin_tdsel_range),
	[PINCTRL_PIN_REG_RDSEL] = MTK_RANGE(mt6765_pin_rdsel_range),
	[PINCTRL_PIN_REG_DRV] = MTK_RANGE(mt6765_pin_drv_range),
	[PINCTRL_PIN_REG_PUPD] = MTK_RANGE(mt6765_pin_pupd_range),
	[PINCTRL_PIN_REG_R0] = MTK_RANGE(mt6765_pin_r0_range),
	[PINCTRL_PIN_REG_R1] = MTK_RANGE(mt6765_pin_r1_range),
	[PINCTRL_PIN_REG_IES] = MTK_RANGE(mt6765_pin_ies_range),
};

static const char * const mt6765_pinctrl_register_base_names[] = {
	"iocfg0", "iocfg1", "iocfg2", "iocfg3", "iocfg4", "iocfg5",
	"iocfg6", "iocfg7",
};

static const struct mtk_eint_hw mt6765_eint_hw = {
	.port_mask = 7,
	.ports     = 6,
	.ap_num    = 160,
	.db_cnt    = 13,
	.db_time   = debounce_time_mt6765,
};

static const struct mtk_pin_soc mt6765_data = {
	.reg_cal = mt6765_reg_cals,
	.pins = mtk_pins_mt6765,
	.npins = ARRAY_SIZE(mtk_pins_mt6765),
	.ngrps = ARRAY_SIZE(mtk_pins_mt6765),
	.eint_hw = &mt6765_eint_hw,
	.gpio_m = 0,
	.base_names = mt6765_pinctrl_register_base_names,
	.nbase_names = ARRAY_SIZE(mt6765_pinctrl_register_base_names),
	.bias_set_combo = mtk_pinconf_bias_set_combo,
	.bias_get_combo = mtk_pinconf_bias_get_combo,
	.drive_set = mtk_pinconf_drive_set_raw,
	.drive_get = mtk_pinconf_drive_get_raw,
	.adv_pull_get = mtk_pinconf_adv_pull_get,
	.adv_pull_set = mtk_pinconf_adv_pull_set,
};

static const struct of_device_id mt6765_pinctrl_of_match[] = {
	{ .compatible = "mediatek,mt6765-pinctrl", .data = &mt6765_data },
	{ }
};
MODULE_DEVICE_TABLE(of, mt6765_pinctrl_of_match);

static struct platform_driver mt6765_pinctrl_driver = {
	.driver = {
		.name = "mt6765-pinctrl",
		.of_match_table = mt6765_pinctrl_of_match,
	},
	.probe = mtk_paris_pinctrl_probe,
};

static int __init mt6765_pinctrl_init(void)
{
	return platform_driver_register(&mt6765_pinctrl_driver);
}
arch_initcall(mt6765_pinctrl_init);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("MediaTek MT6765 Pinctrl Driver");
